/*
 * Decompiled with CFR 0.152.
 */
package net.runelite.client.plugins.gpu;

import com.google.common.base.Charsets;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;
import javax.inject.Singleton;
import net.runelite.client.RuneLiteProperties;
import net.runelite.client.plugins.gpu.GLBuffer;
import net.runelite.client.plugins.gpu.template.Template;
import net.runelite.client.util.OSType;
import net.runelite.rlawt.AWTContext;
import org.jocl.CL;
import org.jocl.CLException;
import org.jocl.NativePointerObject;
import org.jocl.Pointer;
import org.jocl.Sizeof;
import org.jocl.cl_command_queue;
import org.jocl.cl_context;
import org.jocl.cl_context_properties;
import org.jocl.cl_device_id;
import org.jocl.cl_event;
import org.jocl.cl_kernel;
import org.jocl.cl_mem;
import org.jocl.cl_platform_id;
import org.jocl.cl_program;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
class OpenCLManager {
    private static final Logger log = LoggerFactory.getLogger(OpenCLManager.class);
    private static final String GL_SHARING_PLATFORM_EXT = "cl_khr_gl_sharing";
    private static final String KERNEL_NAME_UNORDERED = "computeUnordered";
    private static final String KERNEL_NAME_LARGE = "computeLarge";
    private static final int MIN_WORK_GROUP_SIZE = 256;
    private static final int SMALL_SIZE = 512;
    private static final int LARGE_SIZE = 6144;
    private static final int SHARED_SIZE = 43;
    private int largeFaceCount;
    private int smallFaceCount;
    private cl_platform_id platform;
    private cl_device_id device;
    cl_context context;
    private cl_command_queue commandQueue;
    private cl_program programUnordered;
    private cl_program programSmall;
    private cl_program programLarge;
    private cl_kernel kernelUnordered;
    private cl_kernel kernelSmall;
    private cl_kernel kernelLarge;

    OpenCLManager() {
    }

    void init(AWTContext awtContext) {
        CL.setExceptionsEnabled(true);
        switch (OSType.getOSType()) {
            case Windows: 
            case Linux: {
                this.initPlatform();
                this.initDevice();
                this.initContext(awtContext);
                break;
            }
            case MacOS: {
                this.initMacOS(awtContext);
                break;
            }
            default: {
                throw new RuntimeException("Unsupported OS Type " + OSType.getOSType().name());
            }
        }
        this.ensureMinWorkGroupSize();
        this.initQueue();
        this.compilePrograms();
    }

    void cleanup() {
        if (this.programUnordered != null) {
            CL.clReleaseProgram(this.programUnordered);
            this.programUnordered = null;
        }
        if (this.programSmall != null) {
            CL.clReleaseProgram(this.programSmall);
            this.programSmall = null;
        }
        if (this.programLarge != null) {
            CL.clReleaseProgram(this.programLarge);
            this.programLarge = null;
        }
        if (this.kernelUnordered != null) {
            CL.clReleaseKernel(this.kernelUnordered);
            this.kernelUnordered = null;
        }
        if (this.kernelSmall != null) {
            CL.clReleaseKernel(this.kernelSmall);
            this.kernelSmall = null;
        }
        if (this.kernelLarge != null) {
            CL.clReleaseKernel(this.kernelLarge);
            this.kernelLarge = null;
        }
        if (this.commandQueue != null) {
            CL.clReleaseCommandQueue(this.commandQueue);
            this.commandQueue = null;
        }
        if (this.context != null) {
            CL.clReleaseContext(this.context);
            this.context = null;
        }
        if (this.device != null) {
            CL.clReleaseDevice(this.device);
            this.device = null;
        }
    }

    private String logPlatformInfo(cl_platform_id platform, int param) {
        long[] size = new long[1];
        CL.clGetPlatformInfo(platform, param, 0L, null, size);
        byte[] buffer = new byte[(int)size[0]];
        CL.clGetPlatformInfo(platform, param, buffer.length, Pointer.to(buffer), null);
        String platformInfo = new String(buffer, Charsets.UTF_8);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Platform: {}, {}", (Object)CL.stringFor_cl_platform_info(param), (Object)platformInfo);
        }
        return platformInfo;
    }

    private void logBuildInfo(cl_program program, int param) {
        long[] size = new long[1];
        CL.clGetProgramBuildInfo(program, this.device, param, 0L, null, size);
        ByteBuffer buffer = ByteBuffer.allocateDirect((int)size[0]);
        CL.clGetProgramBuildInfo(program, this.device, param, buffer.limit(), Pointer.toBuffer(buffer), null);
        switch (param) {
            case 4481: {
                if (!RuneLiteProperties.DEBUG_MODE.booleanValue()) break;
                log.debug("Build status: {}, {}", (Object)CL.stringFor_cl_program_build_info(param), (Object)CL.stringFor_cl_build_status(buffer.getInt()));
                break;
            }
            case 4484: {
                if (!RuneLiteProperties.DEBUG_MODE.booleanValue()) break;
                log.debug("Binary type: {}, {}", (Object)CL.stringFor_cl_program_build_info(param), (Object)CL.stringFor_cl_program_binary_type(buffer.getInt()));
                break;
            }
            case 4483: {
                String buildLog = StandardCharsets.US_ASCII.decode(buffer).toString();
                log.trace("Build log: {}, {}", (Object)CL.stringFor_cl_program_build_info(param), (Object)buildLog);
                break;
            }
            case 4482: {
                String message = StandardCharsets.US_ASCII.decode(buffer).toString();
                if (!RuneLiteProperties.DEBUG_MODE.booleanValue()) break;
                log.debug("Build options: {}, {}", (Object)CL.stringFor_cl_program_build_info(param), (Object)message);
                break;
            }
            default: {
                throw new IllegalArgumentException();
            }
        }
    }

    private void initPlatform() {
        int[] platformCount = new int[1];
        CL.clGetPlatformIDs(0, null, platformCount);
        if (platformCount[0] == 0) {
            throw new RuntimeException("No compute platforms found");
        }
        cl_platform_id[] platforms = new cl_platform_id[platformCount[0]];
        CL.clGetPlatformIDs(platforms.length, platforms, null);
        for (cl_platform_id platform : platforms) {
            if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
                log.debug("Found cl_platform_id {}", (Object)platform);
            }
            this.logPlatformInfo(platform, 2304);
            this.logPlatformInfo(platform, 2305);
            this.logPlatformInfo(platform, 2306);
            this.logPlatformInfo(platform, 2307);
            String[] extensions = this.logPlatformInfo(platform, 2308).split(" ");
            if (!Arrays.stream(extensions).anyMatch(s2 -> s2.equals(GL_SHARING_PLATFORM_EXT))) continue;
            this.platform = platform;
        }
        if (this.platform == null) {
            throw new RuntimeException("Platform does not support OpenGL buffer sharing");
        }
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Selected cl_platform_id {}", (Object)this.platform);
        }
    }

    private void initDevice() {
        int[] deviceCount = new int[1];
        CL.clGetDeviceIDs(this.platform, 4L, 0, null, deviceCount);
        if (deviceCount[0] == 0) {
            throw new RuntimeException("No compute devices found");
        }
        cl_device_id[] devices = new cl_device_id[deviceCount[0]];
        CL.clGetDeviceIDs(this.platform, 4L, devices.length, devices, null);
        for (cl_device_id device : devices) {
            long[] size = new long[1];
            CL.clGetDeviceInfo(device, 4144, 0L, null, size);
            byte[] devInfoBuf = new byte[(int)size[0]];
            CL.clGetDeviceInfo(device, 4144, devInfoBuf.length, Pointer.to(devInfoBuf), null);
            if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
                log.debug("Found cl_device_id: {}", (Object)device);
            }
            if (!RuneLiteProperties.DEBUG_MODE.booleanValue()) continue;
            log.debug("Device extensions: {}", (Object)new String(devInfoBuf, Charsets.UTF_8));
        }
        this.device = devices[0];
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Selected cl_device_id {}", (Object)this.device);
        }
    }

    private void initContext(AWTContext awtContext) {
        cl_context_properties contextProps = new cl_context_properties();
        contextProps.addProperty(4228L, this.platform);
        contextProps.addProperty(8200L, awtContext.getGLContext());
        if (OSType.getOSType() == OSType.Linux) {
            contextProps.addProperty(8202L, awtContext.getGLXDisplay());
        } else if (OSType.getOSType() == OSType.Windows) {
            contextProps.addProperty(8203L, awtContext.getWGLHDC());
        }
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Creating context with props: {}", (Object)contextProps);
        }
        this.context = CL.clCreateContext(contextProps, 1, new cl_device_id[]{this.device}, null, null, null);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Created compute context {}", (Object)this.context);
        }
    }

    private void initMacOS(AWTContext awtContext) {
        long cglContext = awtContext.getGLContext();
        long cglShareGroup = awtContext.getCGLShareGroup();
        log.info("{} {}", (Object)cglContext, (Object)cglShareGroup);
        cl_context_properties contextProps = new cl_context_properties();
        contextProps.addProperty(0x10000000L, cglShareGroup);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Creating context with props: {}", (Object)contextProps);
        }
        this.context = CL.clCreateContext(contextProps, 0, null, null, null, null);
        this.device = new cl_device_id();
        CL.clGetGLContextInfoAPPLE(this.context, cglContext, 0x10000002, Sizeof.cl_device_id, Pointer.to((NativePointerObject)this.device), null);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Got macOS CLGL compute device {}", (Object)this.device);
        }
    }

    private void ensureMinWorkGroupSize() {
        long[] maxWorkGroupSize = new long[1];
        CL.clGetDeviceInfo(this.device, 4100, Sizeof.size_t, Pointer.to(maxWorkGroupSize), null);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Device CL_DEVICE_MAX_WORK_GROUP_SIZE: {}", (Object)maxWorkGroupSize[0]);
        }
        if (maxWorkGroupSize[0] < 256L) {
            throw new RuntimeException("Compute device does not support min work group size 256");
        }
        int groupSize = Integer.MIN_VALUE >>> Integer.numberOfLeadingZeros((int)maxWorkGroupSize[0]);
        this.largeFaceCount = 6144 / Math.min(groupSize, 6144);
        this.smallFaceCount = 512 / Math.min(groupSize, 512);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Face counts: small: {}, large: {}", (Object)this.smallFaceCount, (Object)this.largeFaceCount);
        }
    }

    private void initQueue() {
        long[] l = new long[1];
        CL.clGetDeviceInfo(this.device, 4138, 8L, Pointer.to(l), null);
        this.commandQueue = CL.clCreateCommandQueue(this.context, this.device, l[0] & 1L, null);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Created command_queue {}, properties {}", (Object)this.commandQueue, (Object)(l[0] & 1L));
        }
    }

    private cl_program compileProgram(String programSource) {
        log.trace("Compiling program:\n {}", (Object)programSource);
        cl_program program = CL.clCreateProgramWithSource(this.context, 1, new String[]{programSource}, null, null);
        try {
            CL.clBuildProgram(program, 0, null, null, null, null);
        }
        catch (CLException e) {
            this.logBuildInfo(program, 4483);
            throw e;
        }
        this.logBuildInfo(program, 4481);
        this.logBuildInfo(program, 4484);
        this.logBuildInfo(program, 4482);
        this.logBuildInfo(program, 4483);
        return program;
    }

    private cl_kernel getKernel(cl_program program, String kernelName) {
        cl_kernel kernel = CL.clCreateKernel(program, kernelName, null);
        if (RuneLiteProperties.DEBUG_MODE.booleanValue()) {
            log.debug("Loaded kernel {} for program {}", (Object)kernelName, (Object)program);
        }
        return kernel;
    }

    private void compilePrograms() {
        Template templateSmall = new Template().addInclude(OpenCLManager.class).add(key -> key.equals("FACE_COUNT") ? "#define FACE_COUNT " + this.smallFaceCount : null);
        Template templateLarge = new Template().addInclude(OpenCLManager.class).add(key -> key.equals("FACE_COUNT") ? "#define FACE_COUNT " + this.largeFaceCount : null);
        String unordered = new Template().addInclude(OpenCLManager.class).load("comp_unordered.cl");
        String small = templateSmall.load("comp.cl");
        String large = templateLarge.load("comp.cl");
        this.programUnordered = this.compileProgram(unordered);
        this.programSmall = this.compileProgram(small);
        this.programLarge = this.compileProgram(large);
        this.kernelUnordered = this.getKernel(this.programUnordered, KERNEL_NAME_UNORDERED);
        this.kernelSmall = this.getKernel(this.programSmall, KERNEL_NAME_LARGE);
        this.kernelLarge = this.getKernel(this.programLarge, KERNEL_NAME_LARGE);
    }

    void compute(int unorderedModels, int smallModels, int largeModels, GLBuffer sceneVertexBuffer, GLBuffer sceneUvBuffer, GLBuffer vertexBuffer, GLBuffer uvBuffer, GLBuffer unorderedBuffer, GLBuffer smallBuffer, GLBuffer largeBuffer, GLBuffer outVertexBuffer, GLBuffer outUvBuffer, GLBuffer uniformBuffer) {
        cl_mem[] glBuffersAll = new cl_mem[]{sceneVertexBuffer.cl_mem, sceneUvBuffer.cl_mem, unorderedBuffer.cl_mem, smallBuffer.cl_mem, largeBuffer.cl_mem, vertexBuffer.cl_mem, uvBuffer.cl_mem, outVertexBuffer.cl_mem, outUvBuffer.cl_mem, uniformBuffer.cl_mem};
        cl_mem[] glBuffers = (cl_mem[])Arrays.stream(glBuffersAll).filter(Objects::nonNull).toArray(cl_mem[]::new);
        cl_event acquireGLBuffers = new cl_event();
        CL.clEnqueueAcquireGLObjects(this.commandQueue, glBuffers.length, glBuffers, 0, null, acquireGLBuffers);
        cl_event[] computeEvents = new cl_event[]{new cl_event(), new cl_event(), new cl_event()};
        int numComputeEvents = 0;
        if (unorderedModels > 0) {
            CL.clSetKernelArg(this.kernelUnordered, 0, Sizeof.cl_mem, unorderedBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 1, Sizeof.cl_mem, sceneVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 2, Sizeof.cl_mem, vertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 3, Sizeof.cl_mem, sceneUvBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 4, Sizeof.cl_mem, uvBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 5, Sizeof.cl_mem, outVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelUnordered, 6, Sizeof.cl_mem, outUvBuffer.ptr());
            CL.clEnqueueNDRangeKernel(this.commandQueue, this.kernelUnordered, 1, null, new long[]{(long)unorderedModels * 6L}, new long[]{6L}, 1, new cl_event[]{acquireGLBuffers}, computeEvents[numComputeEvents++]);
        }
        if (smallModels > 0) {
            CL.clSetKernelArg(this.kernelSmall, 0, 2220L, null);
            CL.clSetKernelArg(this.kernelSmall, 1, Sizeof.cl_mem, smallBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 2, Sizeof.cl_mem, sceneVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 3, Sizeof.cl_mem, vertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 4, Sizeof.cl_mem, sceneUvBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 5, Sizeof.cl_mem, uvBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 6, Sizeof.cl_mem, outVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 7, Sizeof.cl_mem, outUvBuffer.ptr());
            CL.clSetKernelArg(this.kernelSmall, 8, Sizeof.cl_mem, uniformBuffer.ptr());
            CL.clEnqueueNDRangeKernel(this.commandQueue, this.kernelSmall, 1, null, new long[]{smallModels * (512 / this.smallFaceCount)}, new long[]{512 / this.smallFaceCount}, 1, new cl_event[]{acquireGLBuffers}, computeEvents[numComputeEvents++]);
        }
        if (largeModels > 0) {
            CL.clSetKernelArg(this.kernelLarge, 0, 24748L, null);
            CL.clSetKernelArg(this.kernelLarge, 1, Sizeof.cl_mem, largeBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 2, Sizeof.cl_mem, sceneVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 3, Sizeof.cl_mem, vertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 4, Sizeof.cl_mem, sceneUvBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 5, Sizeof.cl_mem, uvBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 6, Sizeof.cl_mem, outVertexBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 7, Sizeof.cl_mem, outUvBuffer.ptr());
            CL.clSetKernelArg(this.kernelLarge, 8, Sizeof.cl_mem, uniformBuffer.ptr());
            CL.clEnqueueNDRangeKernel(this.commandQueue, this.kernelLarge, 1, null, new long[]{(long)largeModels * (long)(6144 / this.largeFaceCount)}, new long[]{6144 / this.largeFaceCount}, 1, new cl_event[]{acquireGLBuffers}, computeEvents[numComputeEvents++]);
        }
        if (numComputeEvents == 0) {
            CL.clEnqueueReleaseGLObjects(this.commandQueue, glBuffers.length, glBuffers, 0, null, null);
        } else {
            CL.clEnqueueReleaseGLObjects(this.commandQueue, glBuffers.length, glBuffers, numComputeEvents, computeEvents, null);
        }
    }

    void finish() {
        CL.clFinish(this.commandQueue);
    }
}

